<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/site_data.php';

$auth = new Auth();
$auth->requireLogin();

$pdo = getDBConnection();
$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = sanitizeInput($_POST['name'] ?? '');
    $title = sanitizeInput($_POST['title'] ?? '');
    $bio = sanitizeInput($_POST['bio'] ?? '');
    $hero_description = sanitizeInput($_POST['hero_description'] ?? '');
    $story_content = sanitizeInput($_POST['story_content'] ?? '');
    $years_experience = (int)($_POST['years_experience'] ?? 0);
    $projects_completed = (int)($_POST['projects_completed'] ?? 0);
    $happy_clients = (int)($_POST['happy_clients'] ?? 0);
    $profile_image = sanitizeInput($_POST['profile_image'] ?? '');
    
    if (empty($name) || empty($title) || empty($bio)) {
        $error = 'Please fill in all required fields.';
    } else {
        try {
            // Check if about section exists
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM about_section");
            $count = $stmt->fetch()['count'];
            
            if ($count > 0) {
                // Update existing record
                $sql = "UPDATE about_section SET name = ?, title = ?, bio = ?, hero_description = ?, story_content = ?, years_experience = ?, projects_completed = ?, happy_clients = ?, profile_image = ? WHERE id = (SELECT id FROM (SELECT id FROM about_section ORDER BY id DESC LIMIT 1) as temp)";
            } else {
                // Insert new record
                $sql = "INSERT INTO about_section (name, title, bio, hero_description, story_content, years_experience, projects_completed, happy_clients, profile_image) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            }
            
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute([$name, $title, $bio, $hero_description, $story_content, $years_experience, $projects_completed, $happy_clients, $profile_image]);
            
            if ($result) {
                $success = 'About section updated successfully!';
            } else {
                $error = 'Error updating about section. Please try again.';
            }
        } catch(PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}

// Get current about data
$about_data = getAboutData();
$user = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>About Section - Admin Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h2>Portfolio Admin</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php"><i>📊</i> Dashboard</a></li>
                <li><a href="about.php" class="active"><i>👤</i> About Section</a></li>
                <li><a href="portfolio.php"><i>🎨</i> Portfolio</a></li>
                <li><a href="contact.php"><i>📞</i> Contact Info</a></li>
                <li><a href="messages.php"><i>💬</i> Messages</a></li>
                <li><a href="newsletter.php"><i>📧</i> Newsletter</a></li>
                <li><a href="settings.php"><i>⚙️</i> Settings</a></li>
                <li><a href="logout.php"><i>🚪</i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <header class="content-header">
                <h1>About Section</h1>
                <div class="user-info">
                    Welcome, <?php echo htmlspecialchars($user['username']); ?>
                </div>
            </header>

            <div class="dashboard-content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <div class="form-container">
                    <h3>Edit About Section Content</h3>
                    <form method="POST" action="">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="name">Full Name *</label>
                                <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($about_data['name'] ?? ''); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="title">Professional Title *</label>
                                <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($about_data['title'] ?? ''); ?>" placeholder="e.g., Graphic Designer" required>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="hero_description">Hero Section Description</label>
                            <textarea id="hero_description" name="hero_description" rows="3" placeholder="Brief description for hero section"><?php echo htmlspecialchars($about_data['hero_description'] ?? ''); ?></textarea>
                        </div>

                        <div class="form-group">
                            <label for="bio">Short Bio *</label>
                            <textarea id="bio" name="bio" rows="4" placeholder="Brief bio for about preview section" required><?php echo htmlspecialchars($about_data['bio'] ?? ''); ?></textarea>
                        </div>

                        <div class="form-group">
                            <label for="story_content">Full Story Content</label>
                            <textarea id="story_content" name="story_content" rows="6" placeholder="Detailed story for about page"><?php echo htmlspecialchars($about_data['story_content'] ?? ''); ?></textarea>
                        </div>

                        <div class="form-group">
                            <label for="profile_image">Profile Image URL</label>
                            <input type="url" id="profile_image" name="profile_image" value="<?php echo htmlspecialchars($about_data['profile_image'] ?? ''); ?>" placeholder="https://example.com/image.jpg">
                            <small style="color: #666; font-size: 0.9rem;">Enter the full URL to your profile image</small>
                        </div>

                        <h4 style="margin: 2rem 0 1rem 0; color: #333;">Statistics</h4>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="years_experience">Years of Experience</label>
                                <input type="number" id="years_experience" name="years_experience" value="<?php echo $about_data['years_experience'] ?? 0; ?>" min="0">
                            </div>
                            
                            <div class="form-group">
                                <label for="projects_completed">Projects Completed</label>
                                <input type="number" id="projects_completed" name="projects_completed" value="<?php echo $about_data['projects_completed'] ?? 0; ?>" min="0">
                            </div>
                            
                            <div class="form-group">
                                <label for="happy_clients">Happy Clients</label>
                                <input type="number" id="happy_clients" name="happy_clients" value="<?php echo $about_data['happy_clients'] ?? 0; ?>" min="0">
                            </div>
                        </div>

                        <div style="margin-top: 2rem;">
                            <button type="submit" class="btn btn-primary">Update About Section</button>
                            <a href="../about.php" class="btn btn-secondary" target="_blank">Preview About Page</a>
                        </div>
                    </form>
                </div>

                <!-- Current Information Display -->
                <?php if ($about_data): ?>
                <div class="form-container">
                    <h3>Current About Information</h3>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem;">
                        <div>
                            <div style="margin-bottom: 1rem;">
                                <strong>Name:</strong>
                                <span><?php echo htmlspecialchars($about_data['name']); ?></span>
                            </div>
                            
                            <div style="margin-bottom: 1rem;">
                                <strong>Title:</strong>
                                <span><?php echo htmlspecialchars($about_data['title']); ?></span>
                            </div>
                            
                            <div style="margin-bottom: 1rem;">
                                <strong>Bio:</strong>
                                <p style="margin-top: 0.5rem; color: #666;"><?php echo nl2br(htmlspecialchars($about_data['bio'])); ?></p>
                            </div>
                        </div>
                        
                        <div>
                            <div style="margin-bottom: 1rem;">
                                <strong>Experience:</strong>
                                <span><?php echo $about_data['years_experience']; ?> years</span>
                            </div>
                            
                            <div style="margin-bottom: 1rem;">
                                <strong>Projects:</strong>
                                <span><?php echo $about_data['projects_completed']; ?> completed</span>
                            </div>
                            
                            <div style="margin-bottom: 1rem;">
                                <strong>Clients:</strong>
                                <span><?php echo $about_data['happy_clients']; ?> happy clients</span>
                            </div>
                            
                            <?php if ($about_data['profile_image']): ?>
                            <div style="margin-bottom: 1rem;">
                                <strong>Profile Image:</strong>
                                <div style="margin-top: 0.5rem;">
                                    <img src="<?php echo htmlspecialchars($about_data['profile_image']); ?>" alt="Profile" style="max-width: 150px; max-height: 150px; border-radius: 8px; object-fit: cover;">
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>
