<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$pdo = getDBConnection();
$success = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'delete_subscriber') {
        $subscriber_id = (int)($_POST['subscriber_id'] ?? 0);
        if ($subscriber_id > 0) {
            try {
                $stmt = $pdo->prepare("DELETE FROM newsletter_subscribers WHERE id = ?");
                if ($stmt->execute([$subscriber_id])) {
                    $success = 'Subscriber deleted successfully!';
                } else {
                    $error = 'Error deleting subscriber.';
                }
            } catch(PDOException $e) {
                $error = 'Database error: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'update_status') {
        $subscriber_id = (int)($_POST['subscriber_id'] ?? 0);
        $status = $_POST['status'] ?? '';
        if ($subscriber_id > 0 && in_array($status, ['active', 'inactive', 'unsubscribed'])) {
            try {
                $stmt = $pdo->prepare("UPDATE newsletter_subscribers SET status = ? WHERE id = ?");
                if ($stmt->execute([$status, $subscriber_id])) {
                    $success = 'Subscriber status updated successfully!';
                } else {
                    $error = 'Error updating subscriber status.';
                }
            } catch(PDOException $e) {
                $error = 'Database error: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'update_newsletter_text') {
        $newsletter_text = sanitizeInput($_POST['newsletter_text'] ?? '');
        if (!empty($newsletter_text)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO site_settings (setting_key, setting_value, setting_type) VALUES ('newsletter_text', ?, 'text') ON DUPLICATE KEY UPDATE setting_value = ?");
                if ($stmt->execute([$newsletter_text, $newsletter_text])) {
                    $success = 'Newsletter text updated successfully!';
                } else {
                    $error = 'Error updating newsletter text.';
                }
            } catch(PDOException $e) {
                $error = 'Database error: ' . $e->getMessage();
            }
        }
    }
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    try {
        $stmt = $pdo->query("SELECT email, status, subscribed_at FROM newsletter_subscribers ORDER BY subscribed_at DESC");
        $subscribers = $stmt->fetchAll();
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="newsletter_subscribers_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Email', 'Status', 'Subscribed Date']);
        
        foreach ($subscribers as $subscriber) {
            fputcsv($output, [
                $subscriber['email'],
                $subscriber['status'],
                date('Y-m-d H:i:s', strtotime($subscriber['subscribed_at']))
            ]);
        }
        
        fclose($output);
        exit;
    } catch(PDOException $e) {
        $error = 'Error exporting data: ' . $e->getMessage();
    }
}

// Get subscribers with pagination
$page = (int)($_GET['page'] ?? 1);
$per_page = 20;
$offset = ($page - 1) * $per_page;
$status_filter = $_GET['status'] ?? '';

try {
    // Build query with optional status filter
    $where_clause = '';
    $params = [];
    if ($status_filter && in_array($status_filter, ['active', 'inactive', 'unsubscribed'])) {
        $where_clause = 'WHERE status = ?';
        $params[] = $status_filter;
    }
    
    // Get total count
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM newsletter_subscribers $where_clause");
    $stmt->execute($params);
    $total_subscribers = $stmt->fetch()['count'];
    $total_pages = ceil($total_subscribers / $per_page);
    
    // Get subscribers for current page
    $stmt = $pdo->prepare("SELECT * FROM newsletter_subscribers $where_clause ORDER BY subscribed_at DESC LIMIT $offset, $per_page");
    $stmt->execute($params);
    $subscribers = $stmt->fetchAll();
    
    // Get statistics
    $stmt = $pdo->query("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_count,
        SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive_count,
        SUM(CASE WHEN status = 'unsubscribed' THEN 1 ELSE 0 END) as unsubscribed_count
        FROM newsletter_subscribers");
    $stats = $stmt->fetch();
    
    // Get current newsletter text
    $stmt = $pdo->prepare("SELECT setting_value FROM site_settings WHERE setting_key = 'newsletter_text'");
    $stmt->execute();
    $newsletter_text = $stmt->fetch()['setting_value'] ?? 'Get design tips and project updates';
    
} catch(PDOException $e) {
    $error = 'Error fetching subscribers: ' . $e->getMessage();
    $subscribers = [];
    $stats = ['total' => 0, 'active_count' => 0, 'inactive_count' => 0, 'unsubscribed_count' => 0];
    $newsletter_text = 'Get design tips and project updates';
}

$user = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Newsletter - Admin Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h2>Portfolio Admin</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php"><i>📊</i> Dashboard</a></li>
                <li><a href="about.php"><i>👤</i> About Section</a></li>
                <li><a href="portfolio.php"><i>🎨</i> Portfolio</a></li>
                <li><a href="contact.php"><i>📞</i> Contact Info</a></li>
                <li><a href="messages.php"><i>💬</i> Messages</a></li>
                <li><a href="newsletter.php" class="active"><i>📧</i> Newsletter</a></li>
                <li><a href="settings.php"><i>⚙️</i> Settings</a></li>
                <li><a href="logout.php"><i>🚪</i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <header class="content-header">
                <h1>Newsletter Management</h1>
                <div class="user-info">
                    Welcome, <?php echo htmlspecialchars($user['username']); ?>
                </div>
            </header>

            <div class="dashboard-content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <!-- Newsletter Settings -->
                <div class="form-container">
                    <h3>Newsletter Settings</h3>
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="update_newsletter_text">
                        <div class="form-group">
                            <label for="newsletter_text">Newsletter Description Text</label>
                            <input type="text" id="newsletter_text" name="newsletter_text" value="<?php echo htmlspecialchars($newsletter_text); ?>" placeholder="e.g., Get design tips and project updates">
                            <small style="color: #666; font-size: 0.9rem;">This text appears in the newsletter signup section on your website</small>
                        </div>
                        <button type="submit" class="btn btn-primary">Update Newsletter Text</button>
                    </form>
                </div>

                <!-- Statistics -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">📧</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['total']; ?></h3>
                            <p>Total Subscribers</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">✅</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['active_count']; ?></h3>
                            <p>Active Subscribers</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">⏸️</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['inactive_count']; ?></h3>
                            <p>Inactive Subscribers</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">❌</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['unsubscribed_count']; ?></h3>
                            <p>Unsubscribed</p>
                        </div>
                    </div>
                </div>

                <!-- Actions -->
                <div style="margin: 2rem 0; display: flex; gap: 1rem; flex-wrap: wrap;">
                    <a href="?export=csv" class="btn btn-primary">📥 Export CSV</a>
                    <a href="?" class="btn btn-secondary">All Subscribers</a>
                    <a href="?status=active" class="btn btn-secondary">Active Only</a>
                    <a href="?status=inactive" class="btn btn-secondary">Inactive Only</a>
                    <a href="?status=unsubscribed" class="btn btn-secondary">Unsubscribed Only</a>
                </div>

                <!-- Subscribers Table -->
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Subscribed Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($subscribers)): ?>
                                <tr>
                                    <td colspan="4" class="no-data">No subscribers found</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach($subscribers as $subscriber): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($subscriber['email']); ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $subscriber['status']; ?>">
                                                <?php echo ucfirst($subscriber['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M j, Y g:i A', strtotime($subscriber['subscribed_at'])); ?></td>
                                        <td>
                                            <div class="table-actions">
                                                <?php if ($subscriber['status'] !== 'active'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="update_status">
                                                        <input type="hidden" name="subscriber_id" value="<?php echo $subscriber['id']; ?>">
                                                        <input type="hidden" name="status" value="active">
                                                        <button type="submit" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Activate</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <?php if ($subscriber['status'] !== 'inactive'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="update_status">
                                                        <input type="hidden" name="subscriber_id" value="<?php echo $subscriber['id']; ?>">
                                                        <input type="hidden" name="status" value="inactive">
                                                        <button type="submit" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Deactivate</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <?php if ($subscriber['status'] !== 'unsubscribed'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="update_status">
                                                        <input type="hidden" name="subscriber_id" value="<?php echo $subscriber['id']; ?>">
                                                        <input type="hidden" name="status" value="unsubscribed">
                                                        <button type="submit" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Unsubscribe</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this subscriber?')">
                                                    <input type="hidden" name="action" value="delete_subscriber">
                                                    <input type="hidden" name="subscriber_id" value="<?php echo $subscriber['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Delete</button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <div style="margin-top: 2rem; text-align: center;">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <?php 
                        $page_url = "?page=$i";
                        if ($status_filter) $page_url .= "&status=$status_filter";
                        ?>
                        <?php if ($i === $page): ?>
                            <span style="padding: 0.5rem 1rem; background: #8B5A96; color: white; border-radius: 4px; margin: 0 0.25rem;"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="<?php echo $page_url; ?>" style="padding: 0.5rem 1rem; background: #f8f9fa; color: #333; text-decoration: none; border-radius: 4px; margin: 0 0.25rem;"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>
