<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$pdo = getDBConnection();
$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $settings = [
        'site_title' => sanitizeInput($_POST['site_title'] ?? ''),
        'site_description' => sanitizeInput($_POST['site_description'] ?? ''),
        'footer_copyright' => sanitizeInput($_POST['footer_copyright'] ?? ''),
        'hero_title' => sanitizeInput($_POST['hero_title'] ?? ''),
        'hero_subtitle' => sanitizeInput($_POST['hero_subtitle'] ?? ''),
        'favicon_url' => sanitizeInput($_POST['favicon_url'] ?? ''),
        'logo_url' => sanitizeInput($_POST['logo_url'] ?? ''),
        'meta_keywords' => sanitizeInput($_POST['meta_keywords'] ?? ''),
        'google_analytics_id' => sanitizeInput($_POST['google_analytics_id'] ?? ''),
        'contact_form_email' => sanitizeInput($_POST['contact_form_email'] ?? ''),
    ];
    
    $required_fields = ['site_title', 'site_description', 'footer_copyright'];
    $missing_fields = [];
    
    foreach ($required_fields as $field) {
        if (empty($settings[$field])) {
            $missing_fields[] = str_replace('_', ' ', ucfirst($field));
        }
    }
    
    if (!empty($missing_fields)) {
        $error = 'Please fill in all required fields: ' . implode(', ', $missing_fields);
    } else {
        try {
            $pdo->beginTransaction();
            
            foreach ($settings as $key => $value) {
                $stmt = $pdo->prepare("INSERT INTO site_settings (setting_key, setting_value, setting_type) VALUES (?, ?, 'text') ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->execute([$key, $value, $value]);
            }
            
            $pdo->commit();
            $success = 'Site settings updated successfully!';
        } catch(PDOException $e) {
            $pdo->rollBack();
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}

// Get current settings
try {
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM site_settings");
    $settings_data = $stmt->fetchAll();
    
    $current_settings = [];
    foreach ($settings_data as $setting) {
        $current_settings[$setting['setting_key']] = $setting['setting_value'];
    }
} catch(PDOException $e) {
    $error = 'Error fetching settings: ' . $e->getMessage();
    $current_settings = [];
}

$user = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Admin Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h2>Portfolio Admin</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php"><i>📊</i> Dashboard</a></li>
                <li><a href="about.php"><i>👤</i> About Section</a></li>
                <li><a href="portfolio.php"><i>🎨</i> Portfolio</a></li>
                <li><a href="contact.php"><i>📞</i> Contact Info</a></li>
                <li><a href="messages.php"><i>💬</i> Messages</a></li>
                <li><a href="newsletter.php"><i>📧</i> Newsletter</a></li>
                <li><a href="settings.php" class="active"><i>⚙️</i> Settings</a></li>
                <li><a href="logout.php"><i>🚪</i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <header class="content-header">
                <h1>Site Settings</h1>
                <div class="user-info">
                    Welcome, <?php echo htmlspecialchars($user['username']); ?>
                </div>
            </header>

            <div class="dashboard-content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <form method="POST" action="">
                    <!-- General Settings -->
                    <div class="form-container">
                        <h3>General Settings</h3>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="site_title">Site Title *</label>
                                <input type="text" id="site_title" name="site_title" value="<?php echo htmlspecialchars($current_settings['site_title'] ?? ''); ?>" required>
                                <small style="color: #666; font-size: 0.9rem;">Appears in browser title and search results</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="hero_subtitle">Hero Subtitle</label>
                                <input type="text" id="hero_subtitle" name="hero_subtitle" value="<?php echo htmlspecialchars($current_settings['hero_subtitle'] ?? ''); ?>" placeholder="e.g., Graphic Designer">
                                <small style="color: #666; font-size: 0.9rem;">Subtitle displayed on homepage hero section</small>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="site_description">Site Description *</label>
                            <textarea id="site_description" name="site_description" rows="3" required><?php echo htmlspecialchars($current_settings['site_description'] ?? ''); ?></textarea>
                            <small style="color: #666; font-size: 0.9rem;">Meta description for SEO (150-160 characters recommended)</small>
                        </div>

                        <div class="form-group">
                            <label for="hero_title">Hero Title</label>
                            <input type="text" id="hero_title" name="hero_title" value="<?php echo htmlspecialchars($current_settings['hero_title'] ?? ''); ?>" placeholder="e.g., Hi, I'm Manisha">
                            <small style="color: #666; font-size: 0.9rem;">Main title displayed on homepage hero section</small>
                        </div>

                        <div class="form-group">
                            <label for="meta_keywords">Meta Keywords</label>
                            <input type="text" id="meta_keywords" name="meta_keywords" value="<?php echo htmlspecialchars($current_settings['meta_keywords'] ?? ''); ?>" placeholder="graphic design, branding, portfolio">
                            <small style="color: #666; font-size: 0.9rem;">Comma-separated keywords for SEO</small>
                        </div>
                    </div>

                    <!-- Branding Settings -->
                    <div class="form-container">
                        <h3>Branding & Visual</h3>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="logo_url">Logo URL</label>
                                <input type="url" id="logo_url" name="logo_url" value="<?php echo htmlspecialchars($current_settings['logo_url'] ?? ''); ?>" placeholder="https://example.com/logo.png">
                                <small style="color: #666; font-size: 0.9rem;">URL to your site logo image</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="favicon_url">Favicon URL</label>
                                <input type="url" id="favicon_url" name="favicon_url" value="<?php echo htmlspecialchars($current_settings['favicon_url'] ?? ''); ?>" placeholder="https://example.com/favicon.ico">
                                <small style="color: #666; font-size: 0.9rem;">URL to your site favicon (16x16 or 32x32 pixels)</small>
                            </div>
                        </div>
                    </div>

                    <!-- Footer Settings -->
                    <div class="form-container">
                        <h3>Footer Settings</h3>
                        <div class="form-group">
                            <label for="footer_copyright">Footer Copyright Text *</label>
                            <input type="text" id="footer_copyright" name="footer_copyright" value="<?php echo htmlspecialchars($current_settings['footer_copyright'] ?? ''); ?>" required>
                            <small style="color: #666; font-size: 0.9rem;">Copyright text displayed in website footer</small>
                        </div>
                    </div>

                    <!-- Contact & Communication -->
                    <div class="form-container">
                        <h3>Contact & Communication</h3>
                        <div class="form-group">
                            <label for="contact_form_email">Contact Form Email</label>
                            <input type="email" id="contact_form_email" name="contact_form_email" value="<?php echo htmlspecialchars($current_settings['contact_form_email'] ?? ''); ?>" placeholder="contact@yoursite.com">
                            <small style="color: #666; font-size: 0.9rem;">Email address to receive contact form submissions</small>
                        </div>
                    </div>

                    <!-- Analytics & Tracking -->
                    <div class="form-container">
                        <h3>Analytics & Tracking</h3>
                        <div class="form-group">
                            <label for="google_analytics_id">Google Analytics ID</label>
                            <input type="text" id="google_analytics_id" name="google_analytics_id" value="<?php echo htmlspecialchars($current_settings['google_analytics_id'] ?? ''); ?>" placeholder="G-XXXXXXXXXX">
                            <small style="color: #666; font-size: 0.9rem;">Your Google Analytics measurement ID</small>
                        </div>
                    </div>

                    <div style="margin-top: 2rem;">
                        <button type="submit" class="btn btn-primary">Update Settings</button>
                        <a href="../index.php" class="btn btn-secondary" target="_blank">Preview Website</a>
                    </div>
                </form>

                <!-- Current Settings Display -->
                <div class="form-container">
                    <h3>Current Settings Overview</h3>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem;">
                        <div>
                            <h4 style="color: #8B5A96; margin-bottom: 1rem;">General</h4>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Site Title:</strong>
                                <span><?php echo htmlspecialchars($current_settings['site_title'] ?? 'Not set'); ?></span>
                            </div>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Hero Title:</strong>
                                <span><?php echo htmlspecialchars($current_settings['hero_title'] ?? 'Not set'); ?></span>
                            </div>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Hero Subtitle:</strong>
                                <span><?php echo htmlspecialchars($current_settings['hero_subtitle'] ?? 'Not set'); ?></span>
                            </div>
                        </div>
                        
                        <div>
                            <h4 style="color: #8B5A96; margin-bottom: 1rem;">Branding</h4>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Logo:</strong>
                                <span><?php echo $current_settings['logo_url'] ? 'Set' : 'Not set'; ?></span>
                            </div>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Favicon:</strong>
                                <span><?php echo $current_settings['favicon_url'] ? 'Set' : 'Not set'; ?></span>
                            </div>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Copyright:</strong>
                                <span><?php echo htmlspecialchars($current_settings['footer_copyright'] ?? 'Not set'); ?></span>
                            </div>
                        </div>
                        
                        <div>
                            <h4 style="color: #8B5A96; margin-bottom: 1rem;">Analytics</h4>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Google Analytics:</strong>
                                <span><?php echo $current_settings['google_analytics_id'] ? 'Configured' : 'Not configured'; ?></span>
                            </div>
                            <div style="margin-bottom: 0.5rem;">
                                <strong>Contact Email:</strong>
                                <span><?php echo htmlspecialchars($current_settings['contact_form_email'] ?? 'Not set'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
