// Portfolio-specific JavaScript functionality

document.addEventListener('DOMContentLoaded', function() {
    initPortfolioFilter();
    initLightbox();
    initGalleryAnimations();
});

// Portfolio filter functionality
function initPortfolioFilter() {
    const filterButtons = document.querySelectorAll('.filter-btn');
    const galleryItems = document.querySelectorAll('.gallery-item');

    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            const filter = this.getAttribute('data-filter');
            
            // Update active button
            filterButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Filter gallery items
            filterGalleryItems(galleryItems, filter);
        });
    });
}

// Filter gallery items with animation
function filterGalleryItems(items, filter) {
    items.forEach((item, index) => {
        const category = item.getAttribute('data-category');
        const shouldShow = filter === 'all' || category === filter;
        
        if (shouldShow) {
            // Show item with staggered animation
            setTimeout(() => {
                item.style.display = 'block';
                item.style.opacity = '0';
                item.style.transform = 'translateY(20px) scale(0.9)';
                
                // Trigger animation
                requestAnimationFrame(() => {
                    item.style.transition = 'all 0.4s ease-out';
                    item.style.opacity = '1';
                    item.style.transform = 'translateY(0) scale(1)';
                });
            }, index * 50);
        } else {
            // Hide item
            item.style.transition = 'all 0.3s ease-out';
            item.style.opacity = '0';
            item.style.transform = 'translateY(-20px) scale(0.9)';
            
            setTimeout(() => {
                item.style.display = 'none';
            }, 300);
        }
    });
}

// Lightbox functionality
function initLightbox() {
    const lightbox = document.getElementById('lightbox');
    
    // Close lightbox on escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeLightbox();
        }
    });
    
    // Prevent lightbox content from closing when clicked
    const lightboxContent = document.querySelector('.lightbox-content');
    if (lightboxContent) {
        lightboxContent.addEventListener('click', function(e) {
            e.stopPropagation();
        });
    }
}

// Open lightbox with project details
function openLightbox(category, title, description) {
    const lightbox = document.getElementById('lightbox');
    const lightboxImage = document.getElementById('lightbox-image');
    const lightboxTitle = document.getElementById('lightbox-title');
    const lightboxCategory = document.getElementById('lightbox-category');
    const lightboxDescription = document.getElementById('lightbox-description');
    const lightboxClient = document.getElementById('lightbox-client');
    const lightboxYear = document.getElementById('lightbox-year');
    const lightboxTools = document.getElementById('lightbox-tools');
    
    // Set content based on category and title
    const projectData = getProjectData(category, title);
    
    // Create enhanced placeholder for lightbox
    lightboxImage.innerHTML = createLightboxPlaceholder(category, title);
    lightboxTitle.textContent = title;
    lightboxCategory.textContent = getCategoryDisplayName(category);
    lightboxDescription.textContent = description;
    lightboxClient.textContent = projectData.client;
    lightboxYear.textContent = projectData.year;
    lightboxTools.textContent = projectData.tools;
    
    // Show lightbox with animation
    lightbox.classList.add('active');
    document.body.style.overflow = 'hidden';
    
    // Add entrance animation
    const lightboxContent = lightbox.querySelector('.lightbox-content');
    lightboxContent.style.transform = 'scale(0.8) translateY(50px)';
    lightboxContent.style.opacity = '0';
    
    requestAnimationFrame(() => {
        lightboxContent.style.transition = 'all 0.4s cubic-bezier(0.4, 0, 0.2, 1)';
        lightboxContent.style.transform = 'scale(1) translateY(0)';
        lightboxContent.style.opacity = '1';
    });
}

// Close lightbox
function closeLightbox() {
    const lightbox = document.getElementById('lightbox');
    const lightboxContent = lightbox.querySelector('.lightbox-content');
    
    // Add exit animation
    lightboxContent.style.transition = 'all 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
    lightboxContent.style.transform = 'scale(0.8) translateY(50px)';
    lightboxContent.style.opacity = '0';
    
    setTimeout(() => {
        lightbox.classList.remove('active');
        document.body.style.overflow = '';
    }, 300);
}

// Get project data based on category and title
function getProjectData(category, title) {
    const projectDatabase = {
        'pamphlets': {
            'LIC Advertisement': {
                client: 'LIC India',
                year: '2023',
                tools: 'Adobe Photoshop, Illustrator, InDesign'
            },
            'Insurance Benefits Brochure': {
                client: 'LIC India',
                year: '2023',
                tools: 'Adobe Illustrator, InDesign'
            }
        },
        'events': {
            'Manik & Yamini Wedding Welcome': {
                client: 'Private Client',
                year: '2023',
                tools: 'Adobe Illustrator, Photoshop'
            },
            'Love & Unity Wedding Welcome': {
                client: 'Private Client',
                year: '2023',
                tools: 'Adobe Illustrator, Photoshop'
            }
        },
        'promotions': {
            'Singlas Restaurant Brochure': {
                client: 'Singla\'s Restaurant',
                year: '2023',
                tools: 'Adobe Photoshop, Illustrator, InDesign'
            },
            'Maharana Pratap School Brochure': {
                client: 'Maharana Pratap Public School',
                year: '2023',
                tools: 'Adobe InDesign, Illustrator'
            }
        },
        'brands': {
            'Okaya Electric Scooters Poster': {
                client: 'Okaya Electric',
                year: '2023',
                tools: 'Adobe Photoshop, Illustrator'
            },
            'Kinetics Fit Gym Poster': {
                client: 'Kinetics Fit Unisex Gym',
                year: '2023',
                tools: 'Adobe Photoshop, Illustrator'
            }
        }
    };
    
    return projectDatabase[category]?.[title] || {
        client: 'Various',
        year: '2023',
        tools: 'Adobe Creative Suite'
    };
}

// Create enhanced placeholder for lightbox
function createLightboxPlaceholder(category, title) {
    const placeholders = {
        'pamphlets': {
            'LIC Advertisement': `
                <div class="lightbox-placeholder-content pamphlet-design">
                    <div class="lic-design">
                        <h2>LIC</h2>
                        <h3>Life Insurance Corporation</h3>
                        <div class="insurance-benefits">
                            <div class="benefit">✓ Comprehensive Coverage</div>
                            <div class="benefit">✓ Flexible Premium Options</div>
                            <div class="benefit">✓ Tax Benefits</div>
                            <div class="benefit">✓ Trusted Since 1956</div>
                        </div>
                        <div class="cta-section">
                            <p>Secure Your Family's Future Today</p>
                            <div class="contact-info">Call: 1800-XXX-XXXX</div>
                        </div>
                    </div>
                </div>
            `,
            'Insurance Benefits Brochure': `
                <div class="lightbox-placeholder-content pamphlet-design">
                    <div class="benefits-design">
                        <h2>Insurance Benefits</h2>
                        <div class="benefits-grid">
                            <div class="benefit-card">
                                <div class="icon">🛡️</div>
                                <h4>Protection</h4>
                                <p>Complete family protection</p>
                            </div>
                            <div class="benefit-card">
                                <div class="icon">💰</div>
                                <h4>Savings</h4>
                                <p>Build wealth over time</p>
                            </div>
                            <div class="benefit-card">
                                <div class="icon">📈</div>
                                <h4>Investment</h4>
                                <p>Market-linked returns</p>
                            </div>
                        </div>
                    </div>
                </div>
            `
        },
        'events': {
            'Manik & Yamini Wedding Welcome': `
                <div class="lightbox-placeholder-content event-design">
                    <div class="wedding-design">
                        <div class="decorative-border">
                            <div class="corner-ornament top-left">❀</div>
                            <div class="corner-ornament top-right">❀</div>
                            <div class="welcome-text">
                                <h1>Welcome</h1>
                                <div class="couple-names">
                                    <span class="name">Manik</span>
                                    <span class="ampersand">&</span>
                                    <span class="name">Yamini</span>
                                </div>
                                <p class="event-details">Wedding Celebration</p>
                                <div class="date-venue">
                                    <p>December 15, 2023</p>
                                    <p>Grand Ballroom</p>
                                </div>
                            </div>
                            <div class="corner-ornament bottom-left">❀</div>
                            <div class="corner-ornament bottom-right">❀</div>
                        </div>
                    </div>
                </div>
            `,
            'Love & Unity Wedding Welcome': `
                <div class="lightbox-placeholder-content event-design">
                    <div class="wedding-design romantic">
                        <div class="heart-border">
                            <h1>Celebrating</h1>
                            <h2>Love & Unity</h2>
                            <div class="romantic-elements">
                                <div class="heart">💕</div>
                                <div class="rings">💍</div>
                            </div>
                            <p class="welcome-message">Welcome to our special day</p>
                        </div>
                    </div>
                </div>
            `
        },
        'promotions': {
            'Singlas Restaurant Brochure': `
                <div class="lightbox-placeholder-content promotion-design">
                    <div class="restaurant-design">
                        <h1>Singla's Restaurant</h1>
                        <h2>Authentic Indian Cuisine</h2>
                        <div class="menu-highlights">
                            <div class="dish">🍛 Biryani Special</div>
                            <div class="dish">🥘 Butter Chicken</div>
                            <div class="dish">🍜 Dal Makhani</div>
                            <div class="dish">🫓 Fresh Naan</div>
                        </div>
                        <div class="special-offer">
                            <h3>Special Offer</h3>
                            <p>20% Off on Family Meals</p>
                            <p>Valid till month end</p>
                        </div>
                        <div class="contact-details">
                            <p>📍 Location: City Center</p>
                            <p>📞 Call: +91-XXXXX-XXXXX</p>
                        </div>
                    </div>
                </div>
            `,
            'Maharana Pratap School Brochure': `
                <div class="lightbox-placeholder-content promotion-design">
                    <div class="school-design">
                        <h1>Maharana Pratap</h1>
                        <h2>Public School</h2>
                        <div class="school-motto">Excellence in Education</div>
                        <div class="features">
                            <div class="feature">📚 Quality Education</div>
                            <div class="feature">🎓 Experienced Faculty</div>
                            <div class="feature">🏫 Modern Facilities</div>
                            <div class="feature">🎨 Extra-curricular Activities</div>
                        </div>
                        <div class="admission-info">
                            <h3>Admissions Open</h3>
                            <p>Classes: Nursery to XII</p>
                            <p>Apply Now for Academic Year 2024-25</p>
                        </div>
                    </div>
                </div>
            `
        },
        'brands': {
            'Okaya Electric Scooters Poster': `
                <div class="lightbox-placeholder-content brand-design">
                    <div class="scooter-design">
                        <h1>OKAYA</h1>
                        <h2>Electric Scooters</h2>
                        <div class="tagline">Ride the Future</div>
                        <div class="features-grid">
                            <div class="feature">
                                <div class="icon">🛵</div>
                                <p>Stylish Design</p>
                            </div>
                            <div class="feature">
                                <div class="icon">⚡</div>
                                <p>Fast Charging</p>
                            </div>
                            <div class="feature">
                                <div class="icon">🌱</div>
                                <p>Eco-Friendly</p>
                            </div>
                            <div class="feature">
                                <div class="icon">💰</div>
                                <p>Cost Effective</p>
                            </div>
                        </div>
                        <div class="cta">
                            <p>Starting from ₹65,000</p>
                            <p>Visit nearest showroom</p>
                        </div>
                    </div>
                </div>
            `,
            'Kinetics Fit Gym Poster': `
                <div class="lightbox-placeholder-content brand-design">
                    <div class="gym-design">
                        <h1>KINETICS FIT</h1>
                        <h2>Unisex Gym</h2>
                        <div class="tagline">Transform Your Body</div>
                        <div class="gym-features">
                            <div class="feature">💪 Modern Equipment</div>
                            <div class="feature">🏋️ Personal Training</div>
                            <div class="feature">🧘 Yoga Classes</div>
                            <div class="feature">❤️ Cardio Zone</div>
                        </div>
                        <div class="membership-offer">
                            <h3>Special Membership Offer</h3>
                            <p>Join Now & Get 2 Months FREE</p>
                            <p>Limited Time Offer</p>
                        </div>
                        <div class="contact">
                            <p>📍 Prime Location</p>
                            <p>⏰ Open 24/7</p>
                        </div>
                    </div>
                </div>
            `
        }
    };
    
    return placeholders[category]?.[title] || `
        <div class="lightbox-placeholder-content">
            <h2>${title}</h2>
            <p>Professional design work showcasing creativity and attention to detail.</p>
        </div>
    `;
}

// Get display name for category
function getCategoryDisplayName(category) {
    const categoryNames = {
        'pamphlets': 'Pamphlet Design',
        'events': 'Event Welcome Sign',
        'promotions': 'Promotional Material',
        'brands': 'Brand Poster'
    };
    
    return categoryNames[category] || category;
}

// Gallery animations on scroll
function initGalleryAnimations() {
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.style.opacity = '1';
                    entry.target.style.transform = 'translateY(0)';
                }, index * 100);
            }
        });
    }, observerOptions);
    
    galleryItems.forEach((item, index) => {
        item.style.opacity = '0';
        item.style.transform = 'translateY(30px)';
        item.style.transition = 'all 0.6s cubic-bezier(0.4, 0, 0.2, 1)';
        observer.observe(item);
    });
}

// Add hover effects for gallery items
document.addEventListener('DOMContentLoaded', function() {
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    galleryItems.forEach(item => {
        const overlay = item.querySelector('.gallery-overlay');
        
        item.addEventListener('mouseenter', function() {
            overlay.style.opacity = '1';
            this.style.transform = 'translateY(-10px) scale(1.02)';
        });
        
        item.addEventListener('mouseleave', function() {
            overlay.style.opacity = '0';
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
});

// Keyboard navigation for lightbox
document.addEventListener('keydown', function(e) {
    const lightbox = document.getElementById('lightbox');
    
    if (lightbox.classList.contains('active')) {
        switch(e.key) {
            case 'Escape':
                closeLightbox();
                break;
            case 'ArrowLeft':
                // Future: Navigate to previous image
                break;
            case 'ArrowRight':
                // Future: Navigate to next image
                break;
        }
    }
});

// Export functions for global use
window.openLightbox = openLightbox;
window.closeLightbox = closeLightbox;
